/*                     __                                               *\
**     ________ ___   / /  ___     Scala API                            **
**    / __/ __// _ | / /  / _ |    (c) 2006-2008, LAMP/EPFL             **
**  __\ \/ /__/ __ |/ /__/ __ |    http://scala-lang.org/               **
** /____/\___/_/ |_/____/_/ | |                                         **
**                          |/                                          **
\*                                                                      */

// $Id: JSON.scala 14384 2008-03-13 14:21:50Z michelou $


package scala.util.parsing.json

/** 
 * This object provides a simple interface to the JSON parser class.
 *
 *  @author Derek Chen-Becker <"java"+@+"chen-becker"+"."+"org">
 */
object JSON extends Parser {

  /**
   * Parse the given JSON string and return a list of elements. If the
   * string is a JSON object it will be a list of pairs. If it's a JSON 
   * array it will be be a list of individual elements.
   *
   * @param input the given JSON string.
   * @return      an optional list of of elements. 
   */
  def parse(input: String): Option[List[Any]] =
    phrase(root)(new lexical.Scanner(input)) match {
      case Success(result, _) => Some(result)
      case _ => None
    }

  /**
   * Parse the given JSON string and return either a <code>List[Any]</code>
   * if the JSON string specifies an <code>Array</code>, or a
   * <code>Map[String,Any]</code> if the JSON string specifies an object.
   *
   * @param input the given JSON string.
   * @return      an optional list or map.
   */
  def parseFull(input: String): Option[Any] =
    parse(input) match {
      case Some(data) => Some(resolveType(data))
      case None => None
    }

  /**
   * A utility method to resolve a parsed JSON list into objects or 
   * arrays. See the parse method for details.
   */
  def resolveType(input: List[Any]): Any = {
    var objMap = Map[String, Any]()
    
    if (input.forall { 
      case (key: String, value: List[_]) =>
        objMap += (key -> resolveType(value))
        true
      case _ => false
    }) objMap
    else
      input
  }
  
}
