/*                     __                                               *\
**     ________ ___   / /  ___     Scala API                            **
**    / __/ __// _ | / /  / _ |    (c) 2005-2006, LAMP/EPFL             **
**  __\ \/ /__/ __ |/ /__/ __ |                                         **
** /____/\___/_/ |_/____/_/ | |                                         **
**                          |/                                          **
\*                                                                      */

// $Id: Serializer.scala 8892 2006-10-09 08:11:38Z michelou $

package scala.actors.remote

import java.io.{DataInputStream, DataOutputStream, EOFException, IOException}

abstract class Serializer(val service: Service) {
  def serialize(o: AnyRef): Array[byte]
  def deserialize(a: Array[byte]): AnyRef

  [throws(classOf[IOException])]
  def readBytes(inputStream: DataInputStream): Array[byte] = {
    try {
      val length = inputStream.readInt()
      val bytes = new Array[byte](length)
      inputStream.readFully(bytes, 0, length)
      return bytes
    }
    catch {
      case npe: NullPointerException =>
        throw new EOFException("Connection closed.")
    }
  }

  [throws(classOf[IOException]), throws(classOf[ClassNotFoundException])]
  def readObject(inputStream: DataInputStream): AnyRef = {
    val bytes = readBytes(inputStream)
    deserialize(bytes)
  }

  [throws(classOf[IOException])]
  def writeBytes(outputStream: DataOutputStream, bytes: Array[byte]): unit = {
    val length = bytes.length;
    // original length
    outputStream.writeInt(length)
    outputStream.write(bytes, 0, length)
    outputStream.flush()
  }

  [throws(classOf[IOException])]
  def writeObject(outputStream: DataOutputStream, obj: AnyRef) = {
    val bytes = serialize(obj)
    writeBytes(outputStream, bytes)
  }
}
